<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC_Account_Funds_My_Account
 */
class WC_Account_Funds_My_Account {

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'wp', array( $this, 'topup_handler' ) );
		add_action( 'woocommerce_before_my_account', array( $this, 'my_account' ) );
	}

	/**
	 * Handle top-ups
	 */
	public function topup_handler() {
		if ( isset( $_POST['wc_account_funds_topup'] ) && isset( $_POST['_wpnonce'] ) && wp_verify_nonce( $_POST['_wpnonce'], 'account-funds-topup' ) ) {
			$min          = max( 0, get_option( 'account_funds_min_topup' ) );
			$max          = get_option( 'account_funds_max_topup' );
			$topup_amount = wc_clean( $_POST['topup_amount'] );

			if ( $topup_amount < $min ) {
				wc_add_notice( sprintf( __( 'The minimum amount that can be topped up is %s', 'woocommerce-account-funds' ), wc_price( $min ) ), 'error' );
				return;
			} elseif ( $max && $topup_amount > $max ) {
				wc_add_notice( sprintf( __( 'The maximum amount that can be topped up is %s', 'woocommerce-account-funds' ), wc_price( $max ) ), 'error' );
				return;
			}

			WC()->cart->add_to_cart( wc_get_page_id( 'myaccount' ), true, '', '', array( 'top_up_amount' => $topup_amount ) );
			wp_redirect( get_permalink( wc_get_page_id( 'cart' ) ) );
			exit;
		}
	}

	/**
	 * Show funds on account page
	 */
	public function my_account() {
		$funds = WC_Account_Funds::get_account_funds();

		echo '<h2>'. __( 'Account Funds', 'woocommerce-account-funds' ) .'</h2>';
		echo '<p>'. sprintf( __( 'You currently have <strong>%s</strong> worth of funds in your account.', 'woocommerce-account-funds' ), $funds ) . '</p>';

		if ( 'yes' === get_option( 'account_funds_enable_topup' ) ) {
			$this->my_account_topup();
		} else {
			$this->my_account_products();
		}

		$this->my_account_orders();
	}

	/**
	 * Show top up form
	 */
	private function my_account_topup() {
		?>
		<form method="post">
			<h3><label for="topup_amount"><?php _e( 'Top-up Account Funds', 'woocommerce-account-funds' ); ?></label></h3>
			<p class="form-row form-row-first">
				<input type="number" class="input-text" name="topup_amount" id="topup_amount" step="0.01" value="<?php echo esc_attr( get_option( 'account_funds_min_topup' ) ); ?>" min="<?php echo esc_attr( get_option( 'account_funds_min_topup' ) ); ?>" max="<?php echo esc_attr( get_option( 'account_funds_max_topup' ) ); ?>" />
			</p>
			<p class="form-row">
				<input type="hidden" name="wc_account_funds_topup" value="true" />
				<input type="submit" class="button" value="<?php _e( 'Top-up', 'woocommerce-account-funds' ); ?>" />
			</p>
			<?php wp_nonce_field( 'account-funds-topup' ); ?>
		</form>
		<?php
	}

	/**
	 * Show top up products
	 */
	private function my_account_products() {
		$product_ids = get_posts( array(
			'post_type' => 'product',
			'tax_query' => array(
				array(
					'taxonomy' => 'product_type',
					'field'    => 'slug',
					'terms'    => 'deposit',
				)
			),
			'fields' => 'ids'
		) );
		if ( $product_ids ) {
			echo do_shortcode( '[products ids="' . implode( ',', $product_ids ) . '"]' );
		}
	}

	/**
	 * Show deposits
	 */
	private function my_account_orders() {
		$deposits = get_posts( array(
			'numberposts' => 10,
			'meta_key'    => '_customer_user',
			'meta_value'  => get_current_user_id(),
			'post_type'   => 'shop_order',
			'post_status' => array( 'wc-completed', 'wc-processing', 'wc-on-hold' ),
			'meta_query'  => array(
				array(
					'key'   => '_funds_deposited',
					'value' => '1',
				)
			)
		) );

		if ( $deposits ) {
			?>
			<h2><?php _e( 'Recent Deposits', 'woocommerce-account-funds' ); ?></h2>
			<table class="shop_table my_account_deposits">
				<thead>
					<tr>
						<th class="order-number"><span class="nobr"><?php _e( 'Order', 'woocommerce-account-funds' ); ?></span></th>
						<th class="order-date"><span class="nobr"><?php _e( 'Date', 'woocommerce-account-funds' ); ?></span></th>
						<th class="order-total"><span class="nobr"><?php _e(' Status', 'woocommerce-account-funds' ); ?></span></th>
						<th class="order-status"><span class="nobr"><?php _e( 'Amount Funded', 'woocommerce-account-funds' ); ?></span></th>
					</tr>
				</thead>
				<tbody><?php
					foreach ( $deposits as $deposit ) :
						$order  = new WC_Order();
						$order->populate( $deposit );
						$funded = 0;

						foreach ( $order->get_items() as $item ) {
							$product = $order->get_product_from_item( $item );

							if ( $product->is_type( 'deposit' ) || $product->is_type( 'topup' ) ) {
								$funded += $order->get_line_total( $item );
							}
						}
						?><tr class="order">
						<td class="order-number" data-title="<?php _e( 'Order Number', 'woocommerce-account-funds' ); ?>">
							<a href="<?php echo $order->get_view_order_url(); ?>">
								#<?php echo $order->get_order_number(); ?>
							</a>
						</td>
						<td class="order-date" data-title="<?php _e( 'Date', 'woocommerce-account-funds' ); ?>">
							<time datetime="<?php echo date( 'Y-m-d', strtotime( $order->order_date ) ); ?>" title="<?php echo esc_attr( strtotime( $order->order_date ) ); ?>"><?php echo date_i18n( get_option( 'date_format' ), strtotime( $order->order_date ) ); ?></time>
						</td>
						<td class="order-status" data-title="<?php _e( 'Status', 'woocommerce-account-funds' ); ?>" style="text-align:left; white-space:nowrap;">
							<?php echo wc_get_order_status_name( $order->get_status() ); ?>
						</td>
						<td class="order-total">
							<?php echo wc_price( $funded ); ?>
						</td>
					</tr><?php
					endforeach;
				?></tbody>
			</table>
			<?php
		}
	}
}

new WC_Account_Funds_My_Account();
