<?php
/**
 * Restrict Payment Gateways.
 *
 * @class 	WC_CSP_Restrict_Payment_Gateways
 * @version 1.1.6
 * @author 	SomewhereWarm
 *
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WC_CSP_Restrict_Payment_Gateways extends WC_CSP_Restriction implements WC_CSP_Checkout_Restriction {

	public function __construct() {

		$this->id                       = 'payment_gateways';
		$this->title                    = __( 'Payment Gateways', WC_Conditional_Shipping_Payments::TEXT_DOMAIN );
		$this->description              = __( 'Restrict the available payment gateways based on product-related constraints.', WC_Conditional_Shipping_Payments::TEXT_DOMAIN );
		$this->validation_types         = array( 'checkout' );
		$this->has_admin_product_fields = true;
		$this->supports_multiple        = true;

		$this->has_admin_global_fields  = true;
		$this->method_title             = __( 'Payment Gateway Restrictions', WC_Conditional_Shipping_Payments::TEXT_DOMAIN );

		// filter payment gateways when restrictions apply
		add_filter( 'woocommerce_available_payment_gateways', array( $this, 'exclude_payment_gateways' ) );

		// save global settings
		add_action( 'woocommerce_update_options_restrictions_' . $this->id, array( $this, 'update_global_restriction_data' ) );

		// initialize global settings
		$this->init_form_fields();

		// shows a woocommerce error on the 'woocommerce_review_order_before_cart_contents' hook when payment gateway restrictions apply
		add_action( 'woocommerce_review_order_before_cart_contents', array( $this, 'excluded_payment_gateways_notice' ) );
	}

	/**
	 * Declare 'admin_global_fields' type, generated by 'generate_admin_global_fields_html'.
	 *
	 * @return void
	 */
	function init_form_fields() {

		$this->form_fields = array(
			'admin_global_fields' => array(
				'type' => 'admin_global_fields'
			)
		);
	}

	/**
	 * Generates the 'admin_global_fields' field type, which is based on metaboxes.
	 *
	 * @return string
	 */
	function generate_admin_global_fields_html() {
		?><p>
			<?php echo __( 'Restrict the payment gateways available during checkout when the defined conditions apply. Complex rules can be created by adding multiple restriction instances. Each instance will be evaluated independently.', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ); ?>
		</p><?php

		$this->get_admin_global_metaboxes_html();
	}

	/**
	 * Display admin options.
	 *
	 * @param  int    $index    restriction fields array index
	 * @param  string $options  metabox options
	 * @return string
	 */
	public function get_admin_fields_html( $index, $options = array() ) {

		$description   = '';
		$gateways      = array();
		$show_excluded = false;

		if ( isset( $options[ 'description' ] ) ) {
			$description = $options[ 'description' ];
		}

		if ( isset( $options[ 'gateways' ] ) ) {
			$gateways = $options[ 'gateways' ];
		}

		if ( isset( $options[ 'show_excluded' ] ) && $options[ 'show_excluded' ] === 'yes' ) {
			$show_excluded = true;
		}

		$payment_gateways = WC()->payment_gateways->payment_gateways();

		?>
		<p class="form-field">
			<label>
				<?php _e( 'Short Description', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ); ?>:
			</label>
			<textarea class="short_description" name="restriction[<?php echo $index; ?>][description]" id="restriction_<?php echo $index; ?>_short_description" placeholder="<?php _e( 'Optional short description for this rule&hellip;', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ); ?>" rows="1" cols="20"><?php echo $description; ?></textarea>
		</p>
		<p class="form-field">
			<label><?php _e( 'Exclude Gateways', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ); ?></label>
			<select name="restriction[<?php echo $index; ?>][gateways][]" style="width:80%" class="multiselect <?php echo WC_CSP_Core_Compatibility::is_wc_version_gte_2_3() ? 'wc-enhanced-select' : 'chosen_select'; ?>" multiple="multiple" data-placeholder="<?php _e( 'Select Payment Gateways&hellip;', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ); ?>">
				<?php
					foreach ( $payment_gateways as $key => $val ) {
						echo '<option value="' . esc_attr( $key ) . '" ' . selected( in_array( $key, $gateways ), true, false ) . '>' . $val->get_title() . '</option>';
					}
				?>
			</select>
		</p>
		<p class="form-field">
			<label>
				<?php _e( 'Show Excluded', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ); ?>
			</label>
			<input type="checkbox" class="checkbox show_excluded_in_checkout" name="restriction[<?php echo $index; ?>][show_excluded]" <?php echo $show_excluded ? 'checked="checked"' : ''; ?>>
			<?php echo '<img class="help_tip" data-tip="' . __( 'By default, excluded payment gateways are removed from the list of gateways available during checkout. Select this option if you prefer to show excluded gateways in the checkout options and trigger a restriction notice when customers attempt to complete an order using an excluded gateway.', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ) . '" src="' . WC()->plugin_url() . '/assets/images/help.png" />'; ?>
		</p>
		<?php
	}

	/**
	 * Display a short summary of the restriction's settings.
	 *
	 * @param  array  $options
	 * @return string
	 */
	public function get_options_description( $options ) {

		if ( ! empty( $options[ 'description' ] ) ) {
			return $options[ 'description' ];
		}

		$gateway_descriptions = array();
		$gateways             = array();

		if ( isset( $options[ 'gateways' ] ) ) {
			$gateways = $options[ 'gateways' ];
		}

		$payment_gateways = WC()->payment_gateways->payment_gateways();

		foreach ( $payment_gateways as $key => $val ) {

			if ( in_array( $key, $gateways ) ) {
				$gateway_descriptions[] = $val->get_title();
			}
		}

		return trim( implode( ', ', $gateway_descriptions ), ' ,' );
	}

	/**
	 * Display options on the global Restrictions write-panel.
	 *
	 * @param  int    $index    restriction fields array index
	 * @param  string $options  metabox options
	 * @return string
	 */
	function get_admin_global_fields_html( $index, $options = array() ) {

		$this->get_admin_fields_html( $index, $options );
	}

	/**
	 * Display options on the product Restrictions write-panel.
	 *
	 * @param  int    $index    restriction fields array index
	 * @param  string $options  metabox options
	 * @return string
	 */
	function get_admin_product_fields_html( $index, $options = array() ) {
		?><div class="description">
			<em><?php echo __( 'Restrict the payment gateways available during checkout when an order contains this product.', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ); ?></em>
		</div><?php

		$this->get_admin_fields_html( $index, $options );
	}

	/**
	 * Validate, process and return product options.
	 *
	 * @param  array  $posted_data
	 * @return array
	 */
	public function process_admin_fields( $posted_data ) {

		$processed_data = array();

		$processed_data[ 'gateways' ] = array();

		if ( ! empty( $posted_data[ 'gateways' ] ) ) {
			$processed_data[ 'gateways' ] = array_map( 'stripslashes', $posted_data[ 'gateways' ] );
		} else {
			return false;
		}

		if ( isset( $posted_data[ 'show_excluded' ] ) ) {
			$processed_data[ 'show_excluded' ] = 'yes';
		}

		if ( ! empty( $posted_data[ 'description' ] ) ) {
			$processed_data[ 'description' ] = strip_tags ( stripslashes( $posted_data[ 'description' ] ) );
		}

		return $processed_data;
	}

	/**
	 * Validate, process and return product metabox options.
	 *
	 * @param  array  $posted_data
	 * @return array
	 */
	public function process_admin_product_fields( $posted_data ) {

		$processed_data = $this->process_admin_fields( $posted_data );

		if ( ! $processed_data ) {

			WC_Admin_Meta_Boxes::add_error( sprintf( __( 'Restriction #%s was not saved. Before saving a &quot;Payment Gateways&quot; restriction, remember to add at least one payment gateway to the exclusions list.', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ), $posted_data[ 'index' ] ) );
			return false;
		}

		return $processed_data;
	}

	/**
	 * Validate, process and return global settings.
	 *
	 * @param  array  $posted_data
	 * @return array
	 */
	public function process_admin_global_fields( $posted_data ) {

		$processed_data = $this->process_admin_fields( $posted_data );

		if ( ! $processed_data ) {

			WC_Admin_Settings::add_error( sprintf( __( 'Restriction #%s was not saved. Before saving a &quot;Payment Gateways&quot; restriction, remember to add at least one payment gateway to the exclusions list.', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ), $posted_data[ 'index' ] ) );
			return false;
		}

		return $processed_data;
	}

	/**
	 * Shows a woocommerce error on the 'woocommerce_review_order_before_cart_contents' hook when payment gateway restrictions apply.
	 *
	 * @return void
	 */
	public function excluded_payment_gateways_notice() {

		if ( defined( 'WOOCOMMERCE_CHECKOUT' ) ) {

			$result = $this->validate_checkout( array() );

			if ( $result->has_messages() ) {
				foreach ( $result->get_messages() as $message ) {
					wc_add_notice( $message[ 'text' ], $message[ 'type' ] );
				}
			}
		}
	}

	/**
	 * Filter payment gateways when restrictions apply.
	 *
	 * @param  array  $gateways
	 * @return array
	 */
	public function exclude_payment_gateways( $gateways ) {

		/* ----------------------------------------------------------------- */
		/* Product Restrictions
		/* ----------------------------------------------------------------- */

		if ( is_checkout_pay_page() ) {

			global $wp;

			if ( isset( $wp->query_vars[ 'order-pay' ] ) ) {

				$order_id = $wp->query_vars[ 'order-pay' ];
				$order    = wc_get_order( $order_id );

				if ( $order ) {

					$order_items = $order->get_items( 'line_item' );

					if ( ! empty( $order_items ) ) {

						foreach ( $order_items as $order_item ) {

							$product_restriction_data = $this->get_product_restriction_data( $order_item[ 'product_id' ] );

							if ( ! empty( $product_restriction_data ) ) {

								// Evaluate all restriction sets for the current product
								foreach ( $product_restriction_data as $restriction ) {

									// If gateway exclusions are present and all exclusion conditions apply, remove the excluded gateways
									if ( ! empty( $restriction[ 'gateways' ] ) && $this->check_conditions_apply( $restriction, array( 'order_item_data' => $order_item ) ) ) {

										$restricted_gateways = $restriction[ 'gateways' ];

										foreach ( $restricted_gateways as $gateway_id ) {
											if ( isset( $gateways[ $gateway_id ] ) ) {
												unset( $gateways[ $gateway_id ] );
											}
										}
									}
								}
							}
						}
					}
				}
			}

		} else {

			$cart_contents = WC()->cart->get_cart();

			// Loop cart contents
			if ( ! empty( $cart_contents ) ) {
				foreach ( $cart_contents as $cart_item_key => $cart_item_data ) {

					$product = $cart_item_data[ 'data' ];

					$product_restriction_data = $this->get_product_restriction_data( $product->id );

					if ( ! empty( $product_restriction_data ) ) {

						// Evaluate all restriction sets for the current product
						foreach ( $product_restriction_data as $restriction ) {

							// If gateway exclusions are present and all exclusion conditions apply, remove the excluded gateways
							if ( ! empty( $restriction[ 'gateways' ] ) && ! $this->show_excluded( $restriction ) && $this->check_conditions_apply( $restriction, array( 'cart_item_data' => $cart_item_data ) ) ) {

								$restricted_gateways = $restriction[ 'gateways' ];

								foreach ( $restricted_gateways as $gateway_id ) {
									if ( isset( $gateways[ $gateway_id ] ) ) {
										unset( $gateways[ $gateway_id ] );
									}
								}
							}
						}
					}
				}
			}
		}

		/* ----------------------------------------------------------------- */
		/* Global Restrictions
		/* ----------------------------------------------------------------- */

		$global_restriction_data = $this->get_global_restriction_data();

		if ( ! empty( $global_restriction_data ) ) {

			// Evaluate all restriction sets for the current product
			foreach ( $global_restriction_data as $restriction ) {

				// If gateway exclusions are present and all exclusion conditions apply, remove the excluded gateways
				if ( ! empty( $restriction[ 'gateways' ] ) && ! $this->show_excluded( $restriction ) && $this->check_conditions_apply( $restriction ) ) {

					$restricted_gateways = $restriction[ 'gateways' ];

					foreach ( $restricted_gateways as $gateway_id ) {
						if ( isset( $gateways[ $gateway_id ] ) ) {
							unset( $gateways[ $gateway_id ] );
						}
					}
				}
			}
		}

		return $gateways;
	}

	/**
	 * Show excluded gateways or validate selection only.
	 *
	 * @param  array  $restriction_data
	 * @return boolean
	 */
	private function show_excluded( $restriction_data ) {

		if ( ! is_checkout_pay_page() && ! empty( $restriction_data[ 'show_excluded' ] ) && $restriction_data[ 'show_excluded' ] === 'yes' ) {
			return true;
		}

		return false;
	}

	/**
	 * Validate order checkout and return WC_CSP_Check_Result object.
	 *
	 * @param  array  $posted
	 * @return WC_CSP_Check_Result
	 */
	public function validate_checkout( $posted ) {

		$result = new WC_CSP_Check_Result();

		$product_restrictions_exist = false;

		$cart_contents  = WC()->cart->get_cart();
		$chosen_gateway = WC()->session->get( 'chosen_payment_method' );

		if ( ! $chosen_gateway ) {
			return $result;
		}

		$available_gateways = WC()->payment_gateways->payment_gateways();

		/* ----------------------------------------------------------------- */
		/* Product Restrictions
		/* ----------------------------------------------------------------- */

		if ( ! empty( $cart_contents ) ) {
			foreach ( $cart_contents as $cart_item_key => $cart_item_data ) {

				$product = $cart_item_data[ 'data' ];

				$product_restriction_data = $this->get_product_restriction_data( $product->id );

				if ( ! empty( $product_restriction_data ) ) {

					// Evaluate all restriction sets for the current product
					foreach ( $product_restriction_data as $restriction ) {

						$restriction_exists = false;

						// If gateway exclusions are present and all defined restriction conditions apply, add validation error
						if ( ! empty( $restriction[ 'gateways' ] ) && $this->check_conditions_apply( $restriction, array( 'cart_item_data' => $cart_item_data ) ) ) {

							$restricted_gateways = $restriction[ 'gateways' ];

							if ( in_array( $chosen_gateway, $restricted_gateways ) ) {

								$product_restrictions_exist = true;
								$restriction_exists         = true;
							}

						}

						if ( $restriction_exists ) {

							$conditions_resolution = $this->get_conditions_resolution( $restriction, array( 'cart_item_data' => $cart_item_data ) );

							if ( $conditions_resolution ) {
								$resolution = sprintf( __( 'To purchase &quot;%3$s&quot; via &quot;%1$s&quot;, please %2$s. Otherwise, select an alternative payment method, or remove &quot;%3$s&quot; from your cart.', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ), $available_gateways[ $chosen_gateway ]->get_title(), $conditions_resolution, $product->get_title() );
							} else {
								$resolution = sprintf( __( 'To complete your order, please select an alternative payment method, or remove &quot;%1$s&quot; from your cart.', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ), $product->get_title() );
							}

							$message = __( 'Unfortunately, &quot;%1$s&quot; cannot be checked out via &quot;%2$s&quot;. %3$s', WC_Conditional_Shipping_Payments::TEXT_DOMAIN );

							$result->add( 'payment_gateway_excluded_by_product_restriction', sprintf( $message, $product->get_title(), $available_gateways[ $chosen_gateway ]->get_title(), $resolution ) );
						}

					}

				}

			}
		}

		/* ----------------------------------------------------------------- */
		/* Global Restrictions
		/* ----------------------------------------------------------------- */

		$global_restrictions_exist = false;

		// Grab global restrictions
		$global_restriction_data = $this->get_global_restriction_data();

		if ( ! empty( $global_restriction_data ) ) {

			// Evaluate all restriction sets for the current cart
			foreach ( $global_restriction_data as $restriction ) {

				// Check if cart contents not empty
				if ( ! empty( $cart_contents ) ) {

					// Check if non cart-item conditions apply
					if ( ! empty( $restriction[ 'gateways' ] ) && $this->check_conditions_apply( $restriction ) ) {

						$restricted_gateways = $restriction[ 'gateways' ];

						if ( in_array( $chosen_gateway, $restricted_gateways ) ) {

							$global_restrictions_exist = true;

							$conditions_resolution = $this->get_conditions_resolution( $restriction );

							if ( $conditions_resolution ) {
								$resolution = sprintf( __( 'To complete your order via &quot;%1$s&quot;, please %2$s. Otherwise, select an alternative payment method.', WC_Conditional_Shipping_Payments::TEXT_DOMAIN ), $available_gateways[ $chosen_gateway ]->get_title(), $conditions_resolution );
							} else {
								$resolution = __( 'To complete your order, please select an alternative payment method.', WC_Conditional_Shipping_Payments::TEXT_DOMAIN );
							}

							$message = __( 'Unfortunately, your order cannot be checked out via &quot;%1$s&quot;. %2$s', WC_Conditional_Shipping_Payments::TEXT_DOMAIN );

							$result->add( 'payment_gateway_excluded_by_global_restriction', sprintf( $message, $available_gateways[ $chosen_gateway ]->get_title(), $resolution ) );
						}

					}
				}
			}
		}

		return $result;
	}

}
